const BRIDGE_URL = 'http://127.0.0.1:19876';
const STORAGE_KEY = 'claude_limits_data';

// ── Message handler ─────────────────────────────────────────────────────
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'CLAUDE_LIMITS_API_RESPONSE':
      handleApiResponse(message);
      break;
    case 'CLAUDE_LIMITS_DOM_SCRAPE':
      handleDomScrape(message);
      break;
    case 'CLAUDE_LIMITS_RATE_HEADERS':
      handleRateHeaders(message);
      break;
    case 'GET_USAGE_DATA':
      getStoredData().then(data => sendResponse(data));
      return true; // async sendResponse
    case 'GET_BRIDGE_STATUS':
      checkBridgeStatus().then(status => sendResponse(status));
      return true;
  }
});

// ── API response handler ────────────────────────────────────────────────
async function handleApiResponse(message) {
  const { url, data } = message;
  const normalized = normalizeApiData(url, data);
  if (normalized && Object.keys(normalized).length > 0) {
    await updateUsageData(normalized);
  }
}

function normalizeApiData(url, data) {
  if (!data || typeof data !== 'object') return null;
  const result = {};

  // Try common response shapes

  // Shape 1: { daily_usage: X, daily_limit: Y, weekly_usage: ..., ... }
  if (data.daily_usage !== undefined || data.weekly_usage !== undefined) {
    if (data.daily_usage !== undefined) {
      result.daily = {
        used: data.daily_usage,
        total: data.daily_limit,
        percentage: data.daily_limit ? (data.daily_usage / data.daily_limit) * 100 : null,
        resetAt: data.daily_reset_at || data.daily_reset,
      };
    }
    if (data.weekly_usage !== undefined) {
      result.weekly = {
        used: data.weekly_usage,
        total: data.weekly_limit,
        percentage: data.weekly_limit ? (data.weekly_usage / data.weekly_limit) * 100 : null,
        resetAt: data.weekly_reset_at || data.weekly_reset,
      };
    }
  }

  // Shape 2: { usage: { ... }, limits: { ... } }
  if (data.usage || data.limits) {
    const usage = data.usage || {};
    const limits = data.limits || {};
    for (const period of ['session', 'daily', 'weekly', 'monthly']) {
      if (usage[period] !== undefined || limits[period] !== undefined) {
        result[period] = {
          used: usage[period],
          total: limits[period],
          percentage: limits[period] ? (usage[period] / limits[period]) * 100 : null,
        };
      }
    }
  }

  // Shape 3: { rate_limit: { remaining: X, limit: Y, reset: Z } }
  if (data.rate_limit) {
    const rl = data.rate_limit;
    result.rateLimit = {
      remaining: rl.remaining,
      total: rl.limit,
      percentage: rl.limit ? ((rl.limit - rl.remaining) / rl.limit) * 100 : null,
      resetAt: rl.reset,
    };
  }

  // Shape 4: Array of usage items
  if (Array.isArray(data)) {
    for (const item of data) {
      const type = inferPeriodFromItem(item);
      if (type) {
        result[type] = {
          used: item.used ?? item.usage ?? item.current,
          total: item.limit ?? item.total ?? item.max,
          percentage: item.percentage ?? item.percent,
          resetAt: item.reset_at ?? item.resets_at ?? item.reset,
          label: item.name ?? item.label ?? item.type,
        };
        if (result[type].percentage === undefined && result[type].total) {
          result[type].percentage = (result[type].used / result[type].total) * 100;
        }
      }
    }
  }

  // Shape 5: Nested under a key like 'data', 'result', 'response'
  for (const key of ['data', 'result', 'response', 'body']) {
    if (data[key] && typeof data[key] === 'object') {
      const nested = normalizeApiData(url, data[key]);
      if (nested && Object.keys(nested).length > 0) {
        Object.assign(result, nested);
      }
    }
  }

  // If we found nothing structured, store the raw response for debugging
  if (Object.keys(result).length === 0 && url.includes('/usage')) {
    result._raw = { url, keys: Object.keys(data), sample: JSON.stringify(data).substring(0, 500) };
  }

  return result;
}

function inferPeriodFromItem(item) {
  const text = JSON.stringify(item).toLowerCase();
  if (text.includes('session') || text.includes('5_hour') || text.includes('5-hour')) return 'session';
  if (text.includes('daily') || text.includes('day')) return 'daily';
  if (text.includes('weekly') || text.includes('week')) return 'weekly';
  if (text.includes('monthly') || text.includes('month')) return 'monthly';
  if (text.includes('sonnet')) return 'model_sonnet';
  if (text.includes('opus')) return 'model_opus';
  return null;
}

// ── DOM scrape handler ──────────────────────────────────────────────────
async function handleDomScrape(message) {
  const result = {};

  for (const limit of (message.limits || [])) {
    if (limit.type === 'unknown') continue;
    result[limit.type] = {
      percentage: limit.percentage,
      label: limit.label,
      resetAt: limit.resetAt || null,
      source: limit.source || 'dom',
    };
  }

  // Attach reset info to matching metrics
  for (const ri of (message.resetInfo || [])) {
    const key = ri.type;
    if (key && key !== 'unknown' && result[key]) {
      result[key].resetAt = result[key].resetAt || ri.timeString;
    }
    // Also try matching by label overlap
    for (const [mk, mv] of Object.entries(result)) {
      if (!mv.resetAt && ri.label && mv.label &&
          ri.label.toLowerCase().includes(mv.label.toLowerCase().split(' ')[0])) {
        mv.resetAt = ri.timeString;
      }
    }
  }

  if (message.extraSpend) {
    result.extraSpend = message.extraSpend;
  }

  if (message.rawSections?.length > 0) {
    result.rawSections = message.rawSections;
  }

  if (Object.keys(result).length > 0) {
    await updateUsageData(result);
  }
}

function parseRawSection(text) {
  const result = {};
  const lines = text.split('\n').map(l => l.trim()).filter(Boolean);

  // Look for patterns like "Model Usage  45%" or "Weekly  67% used"
  for (let i = 0; i < lines.length; i++) {
    const line = lines[i];
    const pctMatch = line.match(/([\d.]+)\s*%/);
    if (!pctMatch) continue;

    const pct = parseFloat(pctMatch[1]);
    const context = (lines[i - 1] || '') + ' ' + line + ' ' + (lines[i + 1] || '');
    const ctxLower = context.toLowerCase();

    let type = 'unknown';
    if (ctxLower.includes('session') || ctxLower.includes('5 hour') || ctxLower.includes('5-hour')) type = 'session';
    else if (ctxLower.includes('daily') || ctxLower.includes('day')) type = 'daily';
    else if (ctxLower.includes('weekly') || ctxLower.includes('week')) type = 'weekly';
    else if (ctxLower.includes('sonnet')) type = 'model_sonnet';
    else if (ctxLower.includes('opus')) type = 'model_opus';
    else if (ctxLower.includes('all model') || ctxLower.includes('total')) type = 'total';

    if (type !== 'unknown') {
      result[type] = {
        percentage: pct,
        label: context.substring(0, 150).trim(),
        source: 'raw_section',
      };
    }

    // Check for reset time in nearby lines
    const resetMatch = context.match(/resets?\s+in\s+(.+?)(?:\n|\.|$)/i) ||
      context.match(/(\d+[hm]\s*\d*[hm]?)\s*(?:remaining|left)?/i);
    if (resetMatch && type !== 'unknown') {
      result[type].resetIn = resetMatch[1].trim();
    }
  }

  return Object.keys(result).length > 0 ? result : null;
}

// ── Rate-limit headers handler ──────────────────────────────────────────
async function handleRateHeaders(message) {
  const current = await getStoredData();
  current.rateHeaders = {
    ...(current.rateHeaders || {}),
    ...message.headers,
    _lastSeen: Date.now(),
    _url: message.url,
  };
  await storeData(current);
}

// ── Storage helpers ─────────────────────────────────────────────────────
async function getStoredData() {
  const result = await chrome.storage.local.get(STORAGE_KEY);
  return result[STORAGE_KEY] || {};
}

async function storeData(data) {
  await chrome.storage.local.set({ [STORAGE_KEY]: data });
}

async function updateUsageData(newData) {
  const current = await getStoredData();

  // Deep merge: keep existing keys, overwrite with new ones
  for (const [key, value] of Object.entries(newData)) {
    if (typeof value === 'object' && value !== null && !Array.isArray(value) && current[key]) {
      current[key] = { ...current[key], ...value };
    } else {
      current[key] = value;
    }
  }

  current.lastUpdated = Date.now();
  await storeData(current);
  updateBadge(current);
  sendToBridge(current);
}

// ── Badge ───────────────────────────────────────────────────────────────
function updateBadge(data) {
  const percentages = [];
  for (const key of ['session', 'daily', 'weekly', 'monthly', 'total', 'model_opus', 'model_sonnet', 'extra']) {
    const pct = data[key]?.percentage;
    if (typeof pct === 'number' && !isNaN(pct)) percentages.push(pct);
  }

  if (percentages.length === 0) {
    chrome.action.setBadgeText({ text: '?' });
    chrome.action.setBadgeBackgroundColor({ color: '#6b7280' });
    return;
  }

  const maxPct = Math.max(...percentages);
  chrome.action.setBadgeText({ text: `${Math.round(maxPct)}%` });

  if (maxPct >= 90) {
    chrome.action.setBadgeBackgroundColor({ color: '#ef4444' });
  } else if (maxPct >= 70) {
    chrome.action.setBadgeBackgroundColor({ color: '#f59e0b' });
  } else {
    chrome.action.setBadgeBackgroundColor({ color: '#22c55e' });
  }
}

// ── Bridge communication ────────────────────────────────────────────────
async function sendToBridge(data) {
  try {
    await fetch(`${BRIDGE_URL}/update`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data),
    });
  } catch (e) {
    // Bridge server not running — that's OK, extension still works standalone
  }
}

async function checkBridgeStatus() {
  try {
    const res = await fetch(`${BRIDGE_URL}/status`, { signal: AbortSignal.timeout(2000) });
    if (res.ok) {
      return { connected: true, data: await res.json() };
    }
    return { connected: false, error: `HTTP ${res.status}` };
  } catch (e) {
    return { connected: false, error: e.message };
  }
}

// ── Periodic alarms ─────────────────────────────────────────────────────
chrome.alarms.create('syncBridge', { periodInMinutes: 1 });
chrome.alarms.create('forceScrape', { periodInMinutes: 1.5 });

chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name === 'syncBridge') {
    const data = await getStoredData();
    if (data.lastUpdated) {
      sendToBridge(data);
    }
  }
  if (alarm.name === 'forceScrape') {
    // Tell all Claude.ai tabs to re-scrape their DOM
    const tabs = await chrome.tabs.query({ url: 'https://claude.ai/*' });
    for (const tab of tabs) {
      try {
        chrome.tabs.sendMessage(tab.id, { type: 'FORCE_SCRAPE' });
      } catch (e) {
        // Tab may not have content script loaded
      }
    }
  }
});
